"""Climate entities for Navien Smart heating mats."""

import logging
from typing import Any

from homeassistant.components.climate import (
    ClimateEntity,
    ClimateEntityFeature,
    HVACAction,
    HVACMode,
)
from homeassistant.config_entries import ConfigEntry
from homeassistant.const import UnitOfTemperature
from homeassistant.core import HomeAssistant, callback
from homeassistant.helpers.entity_platform import AddEntitiesCallback
from homeassistant.helpers.update_coordinator import CoordinatorEntity

from .const import DOMAIN
from .coordinator import NavienCoordinator
from .navien_api import NavienAPI

_LOGGER = logging.getLogger(__name__)


async def async_setup_entry(
    hass: HomeAssistant,
    entry: ConfigEntry,
    async_add_entities: AddEntitiesCallback,
) -> None:
    """Set up Navien Smart climate entities."""
    coordinator: NavienCoordinator = hass.data[DOMAIN][entry.entry_id]

    def _create_entities(devices: list[dict]) -> list[NavienClimateEntity]:
        entities: list[NavienClimateEntity] = []
        for device in devices:
            is_double = NavienAPI.is_double_mat(device)
            if is_double:
                entities.append(NavienClimateEntity(coordinator, device, "left"))
                entities.append(NavienClimateEntity(coordinator, device, "right"))
            else:
                entities.append(NavienClimateEntity(coordinator, device, "single"))
        return entities

    # Add initial devices
    async_add_entities(_create_entities(coordinator.devices))

    # Register callback for dynamically discovered devices
    @callback
    def _on_new_devices(new_devices: list[dict]) -> None:
        _LOGGER.info("Adding %d new device(s) to HA", len(new_devices))
        async_add_entities(_create_entities(new_devices))

    coordinator.register_new_device_callback(_on_new_devices)


class NavienClimateEntity(CoordinatorEntity[NavienCoordinator], ClimateEntity):
    """Climate entity for a Navien heating mat zone."""

    _attr_has_entity_name = True
    _attr_hvac_modes = [HVACMode.OFF, HVACMode.HEAT]
    _attr_supported_features = (
        ClimateEntityFeature.TARGET_TEMPERATURE | ClimateEntityFeature.TURN_ON | ClimateEntityFeature.TURN_OFF
    )
    _attr_temperature_unit = UnitOfTemperature.CELSIUS
    _attr_target_temperature_step = 0.5
    _enable_turn_on_off_backwards_compat = False

    def __init__(
        self,
        coordinator: NavienCoordinator,
        device: dict,
        zone: str,
    ) -> None:
        super().__init__(coordinator)
        self._device = device
        self._zone = zone  # "single", "left", "right"

        device_id = device["deviceId"]
        device_seq = device["deviceSeq"]
        heat_range = NavienAPI.get_heat_range(device)

        self._attr_min_temp = heat_range["min"]
        self._attr_max_temp = heat_range["max"]

        # Unique ID
        if zone == "single":
            self._attr_unique_id = f"navien_{device_id}"
        else:
            self._attr_unique_id = f"navien_{device_id}_{zone}"

        # Name
        nickname = NavienAPI.get_device_nickname(device, zone if zone != "single" else None)
        if zone == "left":
            self._attr_name = f"{nickname} (L)"
        elif zone == "right":
            self._attr_name = f"{nickname} (R)"
        else:
            self._attr_name = nickname

        # Device info for HA device registry
        self._attr_device_info = {
            "identifiers": {(DOMAIN, str(device_seq))},
            "name": NavienAPI.get_device_nickname(device),
            "manufacturer": "Navien",
            "model": device.get("modelName", "Unknown"),
        }

    @property
    def _reported(self) -> dict | None:
        """Get the latest reported state from MQTT."""
        return self.coordinator.get_device_status(self._device["deviceId"])

    @property
    def available(self) -> bool:
        """Return True if device is connected."""
        reported = self._reported
        if reported is None:
            return super().available
        return reported.get("connected", False)

    @property
    def _zone_data(self) -> dict:
        """Get zone-specific heater data."""
        reported = self._reported
        if reported is None:
            return {}
        return reported.get("heater", {}).get(self._zone, {})

    @property
    def hvac_mode(self) -> HVACMode:
        reported = self._reported
        if reported is None:
            return HVACMode.OFF
        if reported.get("operationMode") != 1:
            return HVACMode.OFF
        # Double mat: zone disabled → OFF (toggle works)
        if self._zone != "single" and not self._zone_data.get("enable", False):
            return HVACMode.OFF
        return HVACMode.HEAT

    @property
    def hvac_action(self) -> HVACAction | None:
        reported = self._reported
        if reported is None:
            return None
        if reported.get("operationMode") != 1:
            return HVACAction.OFF
        zone_data = self._zone_data
        if not zone_data.get("enable", False):
            return HVACAction.IDLE

        temp = zone_data.get("temperature", {})
        current = temp.get("current")
        target = temp.get("set")
        if current is not None and target is not None and current >= target:
            return HVACAction.IDLE
        return HVACAction.HEATING

    @property
    def current_temperature(self) -> float | None:
        reported = self._reported
        if reported is None:
            return None
        heater = reported.get("heater", {})
        zone_data = heater.get(self._zone, {})
        return zone_data.get("temperature", {}).get("current")

    @property
    def target_temperature(self) -> float | None:
        reported = self._reported
        if reported is None:
            return None
        heater = reported.get("heater", {})
        zone_data = heater.get(self._zone, {})
        return zone_data.get("temperature", {}).get("set")

    async def async_set_hvac_mode(self, hvac_mode: HVACMode) -> None:
        """Set HVAC mode - controls zone enable/disable."""
        on = hvac_mode == HVACMode.HEAT
        _LOGGER.info("Setting zone %s %s", self._zone, "ON" if on else "IDLE")
        try:
            if self._zone == "single":
                await self.coordinator.async_control(
                    self._device, {"operationMode": 1 if on else 0}
                )
            else:
                heat_range = NavienAPI.get_heat_range(self._device)
                if on:
                    # Enable zone: set temp to min + step
                    temp = heat_range["min"] + heat_range["step"]
                else:
                    # Check if the other zone is already idle
                    other = "right" if self._zone == "left" else "left"
                    reported = self._reported or {}
                    other_enabled = (
                        reported.get("heater", {})
                        .get(other, {})
                        .get("enable", False)
                    )
                    if not other_enabled:
                        # Both zones would be idle → master power off
                        _LOGGER.info("Both zones idle, turning master power off")
                        await self.coordinator.async_control(
                            self._device, {"operationMode": 0}
                        )
                        return
                    # Idle zone: set temp below min (enable becomes false)
                    temp = heat_range["min"] - heat_range["step"]
                await self.coordinator.api.set_temperature(
                    self._device, temp, self._zone
                )
        except Exception:
            _LOGGER.exception("Failed to set zone for %s", self._attr_name)
            raise

    async def async_set_temperature(self, **kwargs: Any) -> None:
        """Set target temperature."""
        temp = kwargs.get("temperature")
        if temp is None:
            return
        _LOGGER.info("Setting temp %.1f for %s (zone=%s)", temp, self._attr_name, self._zone)
        try:
            if self._zone == "single":
                heater = {"single": {"enable": True, "temperature": {"set": temp}}}
            else:
                heater = {self._zone: {"enable": True, "temperature": {"set": temp}}}
            await self.coordinator.async_control(
                self._device, {"operationMode": 1, "heater": heater}
            )
        except Exception:
            _LOGGER.exception("Failed to set temperature for %s", self._attr_name)
            raise

    async def async_turn_on(self) -> None:
        await self.async_set_hvac_mode(HVACMode.HEAT)

    async def async_turn_off(self) -> None:
        await self.async_set_hvac_mode(HVACMode.OFF)

    @callback
    def _handle_coordinator_update(self) -> None:
        """Handle updated data from the coordinator."""
        self.async_write_ha_state()
