"""Navien Smart async API client."""

import json
import logging

import aiohttp

from .const import API_URL, LOGIN_URL, USER_AGENT

_LOGGER = logging.getLogger(__name__)


class AuthError(Exception):
    """Authentication error."""


class TokenExpiredError(AuthError):
    """Token expired error."""


class ApiError(Exception):
    """API error."""


class NavienAPI:
    """Async Navien Smart API client."""

    def __init__(self):
        self._session: aiohttp.ClientSession | None = None
        self.access_token: str | None = None
        self.refresh_token_str: str | None = None
        self.user_id: str | None = None
        self.account_seq: int | None = None
        self.user_seq: int | None = None
        self.home_seq: int | None = None
        self.home_nickname: str | None = None
        self.aws_credentials: dict | None = None
        self.devices: list[dict] = []

    async def _ensure_session(self):
        if self._session is None or self._session.closed:
            jar = aiohttp.CookieJar(unsafe=True)
            self._session = aiohttp.ClientSession(cookie_jar=jar)

    async def close(self):
        if self._session and not self._session.closed:
            await self._session.close()

    async def web_login(self, username: str, password: str) -> dict:
        """Step 1: Web login at member.naviensmartcontrol.com."""
        await self._ensure_session()
        async with self._session.post(
            f"{LOGIN_URL}/member/login",
            headers={
                "User-Agent": USER_AGENT,
                "Origin": LOGIN_URL,
                "Referer": f"{LOGIN_URL}/member/login",
                "Content-Type": "application/x-www-form-urlencoded",
            },
            data={"username": username, "password": password},
            allow_redirects=True,
            timeout=aiohttp.ClientTimeout(total=15),
        ) as resp:
            html = await resp.text()

        if "passwordChg" in html:
            _LOGGER.warning("Password change required - postponing")
            async with self._session.post(
                f"{LOGIN_URL}/pwchgLate",
                headers={"Content-Type": "application/json"},
                timeout=aiohttp.ClientTimeout(total=15),
            ):
                pass
            return await self.web_login(username, password)

        if 'id="loginFailPopup" style="display:none;"' in html:
            raise AuthError("Login failed: invalid username or password")

        lines = [l for l in html.split("\n") if "var message = " in l]
        if not lines:
            raise AuthError("Login failed: could not find auth token in response")

        line = lines[0].strip()
        json_str = line[line.index("{") : line.rindex("}") + 1]
        return json.loads(json_str)

    async def token_login(self) -> dict:
        """Step 2: Token login via secured-sign-in."""
        await self._ensure_session()
        async with self._session.post(
            f"{API_URL}/users/secured-sign-in",
            headers={
                "Authorization": self.access_token,
                "Content-Type": "application/json",
            },
            json={"userId": self.user_id, "accountSeq": self.account_seq},
            timeout=aiohttp.ClientTimeout(total=15),
        ) as resp:
            data = await resp.json()

        if data.get("code") != 200:
            if data.get("code") == 407 or resp.status == 401:
                raise TokenExpiredError("Access token expired")
            raise ApiError(f"Token login failed: {data}")
        return data

    async def login(self, username: str, password: str):
        """Full 2-step login flow."""
        _LOGGER.debug("Step 1: Web login")
        login_data = await self.web_login(username, password)

        self.access_token = login_data["accessToken"]
        self.refresh_token_str = login_data["refreshToken"]
        self.user_id = login_data["loginId"]
        self.account_seq = login_data["userSeq"]

        _LOGGER.debug("Step 2: Token login (secured-sign-in)")
        token_data = await self.token_login()

        data = token_data["data"]
        self.user_seq = data["userInfo"]["userSeq"]
        homes = data.get("home", [])
        if homes:
            self.home_seq = homes[0]["homeSeq"]
            self.home_nickname = homes[0].get("nickname")
        self.aws_credentials = data["authInfo"]
        _LOGGER.info(
            "Login success: user=%s, home=%s (homeSeq=%s)",
            self.user_id,
            self.home_nickname,
            self.home_seq,
        )

    async def refresh_access_token(self):
        """Refresh the access token using refresh token."""
        await self._ensure_session()
        async with self._session.post(
            f"{API_URL}/auth/token/refresh",
            headers={"Content-Type": "application/json"},
            json={"refreshToken": self.refresh_token_str},
            timeout=aiohttp.ClientTimeout(total=15),
        ) as resp:
            data = await resp.json()

        if data.get("data"):
            self.access_token = data["data"]["authInfo"]["accessToken"]
            _LOGGER.debug("Access token refreshed")
        else:
            raise AuthError("Token refresh failed - re-login required")

    async def refresh_aws_credentials(self):
        """Refresh AWS credentials by re-calling secured-sign-in."""
        try:
            token_data = await self.token_login()
        except TokenExpiredError:
            await self.refresh_access_token()
            token_data = await self.token_login()
        self.aws_credentials = token_data["data"]["authInfo"]
        _LOGGER.debug("AWS credentials refreshed")

    async def _request(self, method: str, path: str, **kwargs) -> dict:
        """Authenticated API request with auto-retry on 407."""
        await self._ensure_session()
        url = f"{API_URL}{path}"
        headers = kwargs.pop("headers", {})
        headers["Authorization"] = self.access_token

        _LOGGER.debug(
            "HTTP %s %s token=%s...%s",
            method, url,
            self.access_token[:10] if self.access_token else "None",
            self.access_token[-6:] if self.access_token else "",
        )

        async with self._session.request(
            method,
            url,
            headers=headers,
            timeout=aiohttp.ClientTimeout(total=15),
            **kwargs,
        ) as resp:
            _LOGGER.debug("HTTP response status=%s", resp.status)
            data = await resp.json()

        if data.get("code") == 407:
            _LOGGER.warning("Token expired (407), refreshing")
            await self.refresh_access_token()
            headers["Authorization"] = self.access_token
            async with self._session.request(
                method,
                url,
                headers=headers,
                timeout=aiohttp.ClientTimeout(total=15),
                **kwargs,
            ) as resp:
                data = await resp.json()

        return data

    async def get_devices(self) -> list[dict]:
        """Get device list."""
        resp = await self._request(
            "GET",
            "/devices",
            params={"homeSeq": str(self.home_seq), "userSeq": str(self.user_seq)},
        )
        if resp.get("code") == 200 and resp.get("data"):
            self.devices = resp["data"].get("devices", [])
            return self.devices
        raise ApiError(f"Failed to get devices: {resp}")

    async def control_device(self, device: dict, payload: dict | None = None):
        """Send control command to a device."""
        device_seq = device["deviceSeq"]
        device_id = device["deviceId"]
        service_code = device["serviceCode"]
        model_code = int(device.get("modelCode", 0))

        desired = {"event": {"modelCode": model_code}}
        if payload:
            desired["beep"] = True
            desired.update(payload)

        body = {
            "serviceCode": service_code,
            "topic": f"$aws/things/{device_id}/shadow/name/status/update",
            "payload": {"state": {"desired": desired}},
        }

        import json as _json
        _LOGGER.info(
            "CONTROL >>> deviceSeq=%s, body=%s",
            device_seq,
            _json.dumps(body, ensure_ascii=False, default=str),
        )

        resp = await self._request(
            "POST",
            f"/devices/{device_seq}/control",
            params={"homeSeq": str(self.home_seq), "userSeq": str(self.user_seq)},
            json=body,
            headers={"Content-Type": "application/json"},
        )

        _LOGGER.info("CONTROL <<< response=%s", resp)
        return resp

    async def set_power(self, device: dict, on: bool):
        """Set device master power on/off (affects all zones)."""
        return await self.control_device(
            device, {"operationMode": 1 if on else 0}
        )

    async def set_zone_power(self, device: dict, zone: str, on: bool):
        """Set individual zone power on/off for double mats."""
        heat_range = self.get_heat_range(device)
        if on:
            # Enable zone with minimum + step temperature
            temp = heat_range["min"] + heat_range["step"]
            heater = {zone: {"enable": True, "temperature": {"set": temp}}}
        else:
            # Disable zone by setting below minimum temperature
            heater = {zone: {"enable": False, "temperature": {"set": heat_range["min"] - heat_range["step"]}}}

        return await self.control_device(
            device, {"operationMode": 1, "heater": heater}
        )

    async def set_temperature(self, device: dict, temp: float, zone: str | None = None):
        """Set temperature for a specific zone. enable = temp >= min."""
        is_double = self.is_double_mat(device)
        heat_range = self.get_heat_range(device)

        def _zone_data(t):
            return {"enable": t >= heat_range["min"], "temperature": {"set": t}}

        if zone and zone in ("left", "right"):
            heater = {zone: _zone_data(temp)}
        elif is_double:
            heater = {"left": _zone_data(temp), "right": _zone_data(temp)}
        else:
            heater = {"single": _zone_data(temp)}

        enable = temp >= heat_range["min"]
        payload = {"heater": heater}
        if enable:
            payload["operationMode"] = 1
        return await self.control_device(device, payload)

    @staticmethod
    def is_double_mat(device: dict) -> bool:
        """Check if device is a double mat."""
        capacity = (
            device.get("Properties", {})
            .get("registry", {})
            .get("attributes", {})
            .get("mcu", {})
            .get("capacity", 1)
        )
        return capacity == 2

    @staticmethod
    def get_heat_range(device: dict) -> dict:
        """Get heat control range from device properties."""
        funcs = (
            device.get("Properties", {})
            .get("registry", {})
            .get("attributes", {})
            .get("functions", {})
        )
        heat = funcs.get("heatControl", {})
        return {
            "min": heat.get("rangeMin", 28),
            "max": heat.get("rangeMax", 50),
            "step": 0.5,
        }

    @staticmethod
    def get_device_nickname(device: dict, zone: str | None = None) -> str:
        """Get device nickname."""
        nickname = device.get("Properties", {}).get("nickName", {})
        if zone and zone in ("left", "right"):
            side = nickname.get("side", {})
            return side.get(zone, zone)
        return nickname.get("mainItem", device.get("modelName", "Navien"))
