"""Switch entities for Navien Smart integration."""

import logging

from homeassistant.components.switch import SwitchEntity
from homeassistant.config_entries import ConfigEntry
from homeassistant.core import HomeAssistant, callback
from homeassistant.helpers.entity_platform import AddEntitiesCallback
from homeassistant.helpers.update_coordinator import CoordinatorEntity

from .const import DOMAIN
from .coordinator import NavienCoordinator
from .navien_api import NavienAPI

_LOGGER = logging.getLogger(__name__)


async def async_setup_entry(
    hass: HomeAssistant,
    entry: ConfigEntry,
    async_add_entities: AddEntitiesCallback,
) -> None:
    """Set up Navien Smart switch entities."""
    coordinator: NavienCoordinator = hass.data[DOMAIN][entry.entry_id]

    # Only create power switch for double mats (single mat uses climate ON/OFF)
    entities = [
        NavienPowerSwitch(coordinator, d)
        for d in coordinator.devices
        if NavienAPI.is_double_mat(d)
    ]
    async_add_entities(entities)

    @callback
    def _on_new_devices(new_devices: list[dict]) -> None:
        async_add_entities([
            NavienPowerSwitch(coordinator, d)
            for d in new_devices
            if NavienAPI.is_double_mat(d)
        ])

    coordinator.register_new_device_callback(_on_new_devices)


class NavienPowerSwitch(CoordinatorEntity[NavienCoordinator], SwitchEntity):
    """Master power switch for a Navien device."""

    _attr_has_entity_name = True
    _attr_icon = "mdi:power"

    def __init__(self, coordinator: NavienCoordinator, device: dict) -> None:
        super().__init__(coordinator)
        self._device = device
        device_id = device["deviceId"]
        device_seq = device["deviceSeq"]

        self._attr_unique_id = f"navien_{device_id}_power"
        self._attr_name = f"{NavienAPI.get_device_nickname(device)} 전원"
        self._attr_device_info = {
            "identifiers": {(DOMAIN, str(device_seq))},
            "name": NavienAPI.get_device_nickname(device),
            "manufacturer": "Navien",
            "model": device.get("modelName", "Unknown"),
        }

    @property
    def _reported(self) -> dict | None:
        return self.coordinator.get_device_status(self._device["deviceId"])

    @property
    def available(self) -> bool:
        reported = self._reported
        if reported is None:
            return super().available
        return reported.get("connected", False)

    @property
    def is_on(self) -> bool:
        reported = self._reported
        if reported is None:
            return False
        return reported.get("operationMode") == 1

    async def async_turn_on(self, **kwargs) -> None:
        _LOGGER.info("Master power ON for %s", self._attr_name)
        await self.coordinator.async_control(
            self._device, {"operationMode": 1}
        )

    async def async_turn_off(self, **kwargs) -> None:
        _LOGGER.info("Master power OFF for %s", self._attr_name)
        await self.coordinator.async_control(
            self._device, {"operationMode": 0}
        )

    @callback
    def _handle_coordinator_update(self) -> None:
        self.async_write_ha_state()
